/**
   BAREOS® - Backup Archiving REcovery Open Sourced

   Copyright (C) 2018-2020 Bareos GmbH & Co. KG

   This program is Free Software; you can redistribute it and/or
   modify it under the terms of version three of the GNU Affero General Public
   License as published by the Free Software Foundation and included
   in the file LICENSE.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   Affero General Public License for more details.

   You should have received a copy of the GNU Affero General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
   02110-1301, USA.
*/
#if defined(HAVE_MINGW)
#  include "include/bareos.h"
#  include "gtest/gtest.h"
#else
#  include "gtest/gtest.h"
#  include "include/bareos.h"
#endif

#include <array>
#include <numeric>
#include "stored/crc32/crc32.h"


TEST(crc32, shortstring)
{
  static const char* buf = "The quick brown fox jumps over the lazy dog";
  EXPECT_EQ(0x414fa339, crc32_fast((uint8_t*)buf, strlen(buf)));
}

TEST(crc32, long_block)
{
  constexpr size_t len = 63 * 1024;
  std::array<uint8_t, len> buf;
  std::iota(buf.begin(), buf.end(), 0xbb);

  ASSERT_EQ(0xcf6610d8, crc32_fast(buf.data(), buf.size()));
}

TEST(crc32, label_block)
{
  constexpr std::array<uint8_t, 204> label_block{
      0xcb, 0x67, 0x8d, 0xdd, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00, 0x00,
      0x42, 0x42, 0x30, 0x32, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa8,
      0x42, 0x61, 0x72, 0x65, 0x6f, 0x73, 0x20, 0x32, 0x2e, 0x30, 0x20, 0x69,
      0x6d, 0x6d, 0x6f, 0x72, 0x74, 0x61, 0x6c, 0x0a, 0x00, 0x00, 0x00, 0x00,
      0x14, 0x00, 0x05, 0x9d, 0xf7, 0x89, 0x5a, 0x0f, 0xdc, 0x00, 0x05, 0x9d,
      0xf7, 0x89, 0x5a, 0x10, 0x37, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x72, 0x65,
      0x65, 0x62, 0x73, 0x64, 0x2d, 0x31, 0x38, 0x2e, 0x32, 0x2e, 0x37, 0x00,
      0x00, 0x53, 0x63, 0x72, 0x61, 0x74, 0x63, 0x68, 0x00, 0x42, 0x61, 0x63,
      0x6b, 0x75, 0x70, 0x00, 0x46, 0x69, 0x6c, 0x65, 0x00, 0x66, 0x72, 0x65,
      0x65, 0x62, 0x73, 0x64, 0x00, 0x66, 0x72, 0x65, 0x65, 0x62, 0x73, 0x64,
      0x2d, 0x73, 0x64, 0x00, 0x56, 0x65, 0x72, 0x2e, 0x20, 0x31, 0x38, 0x2e,
      0x32, 0x2e, 0x37, 0x20, 0x31, 0x32, 0x20, 0x44, 0x65, 0x63, 0x65, 0x6d,
      0x62, 0x65, 0x72, 0x20, 0x32, 0x30, 0x31, 0x39, 0x00, 0x42, 0x75, 0x69,
      0x6c, 0x64, 0x20, 0x44, 0x65, 0x63, 0x20, 0x31, 0x32, 0x20, 0x32, 0x30,
      0x31, 0x39, 0x20, 0x31, 0x37, 0x3a, 0x30, 0x39, 0x3a, 0x34, 0x31, 0x00};

  /* the first four bytes of a block contain the crc32 and are not part of the
   * checksum, so we skip them
   */
  ASSERT_EQ(0xcb678ddd,
            crc32_fast(label_block.data() + 4, label_block.size() - 4));
}
